<?php
/**
 * KESKA Mali Takip Sistemi
 * Ana Giriş Noktası
 */

// Autoload ve config
require_once __DIR__ . '/../includes/autoload.php';

// Session Auth sınıfı tarafından başlatılacak (gereksiz session_start kaldırıldı)

// Auth instance
$auth = new Auth();

// URL parsing
$requestUri = $_SERVER['REQUEST_URI'];
$scriptName = dirname($_SERVER['SCRIPT_NAME']);
$uri = str_replace($scriptName, '', $requestUri);
$uri = trim($uri, '/');
$uri = explode('?', $uri)[0]; // Query string'i kaldır

// Route parametreleri
$segments = explode('/', $uri);
$controller = $segments[0] ?: 'dashboard';
$action = $segments[1] ?? 'index';
$id = $segments[2] ?? null;

// Public route'lar (login gerektirmez)
$publicRoutes = ['login', 'logout'];

// Login kontrolü
if (!in_array($controller, $publicRoutes)) {
    $auth->requireLogin();
}

// Route mapping
try {
    switch ($controller) {
        case '':
        case 'dashboard':
            require_once __DIR__ . '/../controllers/DashboardController.php';
            $ctrl = new DashboardController();
            $ctrl->index();
            break;

        case 'login':
            require_once __DIR__ . '/../controllers/AuthController.php';
            $ctrl = new AuthController();
            if ($action === 'process') {
                $ctrl->processLogin();
            } else {
                $ctrl->showLogin();
            }
            break;

        case 'logout':
            require_once __DIR__ . '/../controllers/AuthController.php';
            $ctrl = new AuthController();
            $ctrl->logout();
            break;

        case 'suppliers':
            require_once __DIR__ . '/../controllers/SupplierController.php';
            $ctrl = new SupplierController();
            switch ($action) {
                case 'index':
                case '':
                    $ctrl->index();
                    break;
                case 'create':
                    $ctrl->create();
                    break;
                case 'store':
                    $ctrl->store();
                    break;
                case 'edit':
                    $ctrl->edit($id);
                    break;
                case 'update':
                    $ctrl->update($id);
                    break;
                case 'delete':
                    $ctrl->delete($id);
                    break;
                case 'view':
                    $ctrl->view($id);
                    break;
                case 'save-transaction':
                    $ctrl->saveTransaction($id);
                    break;
                case 'delete-transaction':
                    // URL: /suppliers/delete-transaction/{supplierId}/{transactionId}
                    // $id = supplierId, $segments[3] = transactionId
                    $transactionId = $segments[3] ?? null;
                    if ($transactionId) {
                        $ctrl->deleteTransaction($id, $transactionId);
                    } else {
                        http_response_code(404);
                        echo "İşlem ID bulunamadı";
                    }
                    break;
                default:
                    http_response_code(404);
                    echo "Sayfa bulunamadı";
            }
            break;

        case 'banks':
            // BCH (Banka Cari Hesabı) Sub-routing
            if ($action === 'bch') {
                require_once __DIR__ . '/../controllers/BankBchController.php';
                $bchCtrl = new BankBchController();
                $subAction = $id ?? 'index';
                $subId = $segments[3] ?? null;

                switch ($subAction) {
                    case 'index':
                        $bchCtrl->index();
                        break;
                    case 'create':
                        $bchCtrl->create();
                        break;
                    case 'store':
                        $bchCtrl->store();
                        break;
                    case 'edit':
                        $bchCtrl->edit($subId);
                        break;
                    case 'update':
                        $bchCtrl->update($subId);
                        break;
                    case 'mark-paid':
                        $bchCtrl->markAsPaid($subId);
                        break;
                    case 'delete':
                        $bchCtrl->delete($subId);
                        break;
                    default:
                        $bchCtrl->index();
                }
            }
            // KMH Sub-routing
            else if ($action === 'kmh') {
                require_once __DIR__ . '/../controllers/BankKmhController.php';
                $kmhCtrl = new BankKmhController();
                $subAction = $id ?? 'index';
                $subId = $segments[3] ?? null;

                switch ($subAction) {
                    case 'index':
                        $kmhCtrl->index();
                        break;
                    case 'create':
                        $kmhCtrl->create();
                        break;
                    case 'store':
                        $kmhCtrl->store();
                        break;
                    case 'view':
                        $kmhCtrl->view($subId);
                        break;
                    case 'edit':
                        $kmhCtrl->edit($subId);
                        break;
                    case 'update':
                        $kmhCtrl->update($subId);
                        break;
                    case 'save-transaction':
                        $kmhCtrl->saveTransaction($subId);
                        break;
                    case 'delete-transaction':
                        $transactionId = $segments[4] ?? null;
                        if ($subId && $transactionId) {
                            $kmhCtrl->deleteTransaction($subId, $transactionId);
                        } else {
                            http_response_code(404);
                            echo "İşlem ID bulunamadı";
                        }
                        break;
                    case 'delete':
                        $kmhCtrl->delete($subId);
                        break;
                    default:
                        $kmhCtrl->index();
                }
            }
            // Installment Loans Sub-routing
            else if ($action === 'installment-loans') {
                require_once __DIR__ . '/../controllers/BankInstallmentLoanController.php';
                $installmentCtrl = new BankInstallmentLoanController();
                $subAction = $id ?? 'index';
                $subId = $segments[3] ?? null;
                $subExtraId = $segments[4] ?? null;

                switch ($subAction) {
                    case 'index':
                        $installmentCtrl->index();
                        break;
                    case 'store':
                        $installmentCtrl->store();
                        break;
                    case 'update':
                        $installmentCtrl->update($subId);
                        break;
                    case 'mark-paid':
                        $installmentCtrl->markPaid($subId, $subExtraId);
                        break;
                    case 'mark-pending':
                        $installmentCtrl->markPending($subId, $subExtraId);
                        break;
                    case 'delete':
                        $installmentCtrl->delete($subId);
                        break;
                    default:
                        $installmentCtrl->index();
                }
            }
            // Default: Main banks overview
            else {
                require_once __DIR__ . '/../controllers/BankController.php';
                $ctrl = new BankController();
                switch ($action) {
                    case 'view':
                        $ctrl->view($id);
                        break;
                    case 'save-transaction':
                        $ctrl->saveTransaction($id);
                        break;
                    case 'delete-transaction':
                        $transactionId = $segments[3] ?? null;
                        if ($transactionId) {
                            $ctrl->deleteTransaction($id, $transactionId);
                        }
                        break;
                    default:
                        $ctrl->index();
                }
            }
            break;

        case 'credit-cards':
            require_once __DIR__ . '/../controllers/CreditCardController.php';
            $ctrl = new CreditCardController();
            $ctrl->$action($id);
            break;

        case 'monthly-payments':
            require_once __DIR__ . '/../controllers/MonthlyPaymentController.php';
            $ctrl = new MonthlyPaymentController();
            $ctrl->$action($id);
            break;

        case 'personnel-attendance':
            require_once __DIR__ . '/../controllers/PersonnelAttendanceController.php';
            $ctrl = new PersonnelAttendanceController();
            $ctrl->$action($id);
            break;

        case 'personnel-salary':
            require_once __DIR__ . '/../controllers/PersonnelSalaryController.php';
            $ctrl = new PersonnelSalaryController();
            $ctrl->$action($id);
            break;

        case 'personnel-card':
            require_once __DIR__ . '/../controllers/PersonnelCardController.php';
            $ctrl = new PersonnelCardController();
            $ctrl->generate();
            break;

        case 'personnel-settings':
            require_once __DIR__ . '/../controllers/PersonnelSettingsController.php';
            $ctrl = new PersonnelSettingsController();
            $ctrl->$action($id);
            break;

        case 'calendar':
            require_once __DIR__ . '/../controllers/CalendarController.php';
            $ctrl = new CalendarController();
            $ctrl->index();
            break;

        case 'all-payments':
            require_once __DIR__ . '/../controllers/AllPaymentsController.php';
            $ctrl = new AllPaymentsController();
            $ctrl->index();
            break;

        case 'checks':
            require_once __DIR__ . '/../controllers/CheckController.php';
            $ctrl = new CheckController();
            switch ($action) {
                case 'index':
                case '':
                    $ctrl->index();
                    break;
                // Verilen Çek İşlemleri
                case 'mark-verilen-collected':
                    $ctrl->markVerilenCollected($id);
                    break;
                case 'mark-verilen-pending':
                    $ctrl->markVerilenPending($id);
                    break;
                // Alınan Çek İşlemleri
                case 'mark-alinan-collected':
                    $ctrl->markAlinanCollected($id);
                    break;
                case 'mark-alinan-pending':
                    $ctrl->markAlinanPending($id);
                    break;
                case 'mark-alinan-endorsed':
                    $ctrl->markAlinanEndorsed($id);
                    break;
                case 'delete-alinan':
                    $ctrl->deleteAlinan($id);
                    break;
                default:
                    $ctrl->index();
            }
            break;

        case 'supplier-checks':
            require_once __DIR__ . '/../controllers/SupplierCheckController.php';
            $ctrl = new SupplierCheckController();
            switch ($action) {
                case 'index':
                case '':
                    $ctrl->index();
                    break;
                case 'mark-collected':
                    $ctrl->markCollected($id);
                    break;
                case 'mark-pending':
                    $ctrl->markPending($id);
                    break;
                default:
                    $ctrl->index();
            }
            break;

        case 'received-checks':
            require_once __DIR__ . '/../controllers/ReceivedCheckController.php';
            $ctrl = new ReceivedCheckController();
            switch ($action) {
                case 'index':
                case '':
                    $ctrl->index();
                    break;
                case 'create':
                    $ctrl->create();
                    break;
                case 'store':
                    $ctrl->store();
                    break;
                case 'mark-collected':
                    $ctrl->markCollected($id);
                    break;
                case 'mark-pending':
                    $ctrl->markPending($id);
                    break;
                case 'mark-endorsed':
                    $ctrl->markEndorsed($id);
                    break;
                case 'delete':
                    $ctrl->delete($id);
                    break;
                default:
                    $ctrl->index();
            }
            break;

        case 'check-import':
            require_once __DIR__ . '/../controllers/CheckImportController.php';
            $ctrl = new CheckImportController();
            if ($action === 'process') {
                $ctrl->importChecks();
            } else {
                http_response_code(404);
                echo "Sayfa bulunamadı";
            }
            break;

        case 'import':
            require_once __DIR__ . '/../controllers/ImportController.php';
            $ctrl = new ImportController();
            if ($action === 'process') {
                $ctrl->process();
            } else {
                $ctrl->index();
            }
            break;

        case 'api':
            // API endpoint'leri
            require_once __DIR__ . '/../controllers/ApiController.php';
            $ctrl = new ApiController();
            $ctrl->handle($action, $id);
            break;

        default:
            http_response_code(404);
            echo "Sayfa bulunamadı: " . htmlspecialchars($controller);
            break;
    }
} catch (Exception $e) {
    if (APP_DEBUG) {
        echo "<h1>Hata</h1>";
        echo "<p>" . $e->getMessage() . "</p>";
        echo "<pre>" . $e->getTraceAsString() . "</pre>";
    } else {
        echo "Bir hata oluştu. Lütfen daha sonra tekrar deneyin.";
        error_log($e->getMessage());
    }
}
