<?php
/**
 * Helper Fonksiyonları
 * Yardımcı ve genel amaçlı fonksiyonlar
 */

class Helper {

    /**
     * XSS koruması
     */
    public static function escape($data) {
        if (is_array($data)) {
            return array_map([self::class, 'escape'], $data);
        }
        return htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    }

    /**
     * URL yönlendirme
     */
    public static function redirect($path = '') {
        $url = BASE_URL . '/' . ltrim($path, '/');
        header("Location: $url");
        exit;
    }

    /**
     * Flash mesaj ayarla
     */
    public static function setFlash($type, $message) {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        $_SESSION['flash'] = [
            'type' => $type,
            'message' => $message
        ];
    }

    /**
     * Flash mesaj al
     */
    public static function getFlash() {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }

        if (isset($_SESSION['flash'])) {
            $flash = $_SESSION['flash'];
            unset($_SESSION['flash']);
            return $flash;
        }

        return null;
    }

    /**
     * Para formatı (Türk Lirası)
     */
    public static function formatMoney($amount, $showCurrency = true) {
        $formatted = number_format($amount, 2, ',', '.');
        return $showCurrency ? $formatted . ' ₺' : $formatted;
    }

    /**
     * Tarih formatı (Türkiye)
     */
    public static function formatDate($date, $format = 'd.m.Y') {
        if (!$date) return '';
        return date($format, strtotime($date));
    }

    /**
     * Tarih ve saat formatı
     */
    public static function formatDateTime($datetime, $format = 'd.m.Y H:i') {
        if (!$datetime) return '';
        return date($format, strtotime($datetime));
    }

    /**
     * MySQL tarih formatına çevir
     */
    public static function toMysqlDate($date) {
        if (!$date) return null;
        return date('Y-m-d', strtotime($date));
    }

    /**
     * Durum badge rengi
     */
    public static function getStatusBadge($status) {
        $badges = [
            'pending' => 'warning',
            'paid' => 'success',
            'overdue' => 'danger',
            'payment_requested' => 'info',
            'partial' => 'secondary',
            'uncertain' => 'secondary'
        ];

        return $badges[$status] ?? 'secondary';
    }

    /**
     * Durum Türkçe çevirisi
     */
    public static function translateStatus($status) {
        $translations = [
            'pending' => 'Beklemede',
            'paid' => 'Ödendi',
            'overdue' => 'Gecikmiş',
            'payment_requested' => 'Ödeme Talep Edildi',
            'partial' => 'Kısmi Ödeme',
            'uncertain' => 'Belirsiz'
        ];

        return $translations[$status] ?? $status;
    }

    /**
     * Kategori Türkçe çevirisi
     */
    public static function translateCategory($category) {
        $translations = [
            'kdv' => 'KDV',
            'sgk' => 'SGK',
            'rent' => 'Kira',
            'electricity' => 'Elektrik',
            'water' => 'Su',
            'gas' => 'Doğalgaz',
            'phone' => 'Telefon',
            'internet' => 'İnternet',
            'salary' => 'Maaş',
            'other' => 'Diğer',
            'tax' => 'Vergi',
            'insurance' => 'Sigorta',
            'utilities' => 'Faturalar'
        ];

        return $translations[$category] ?? $category;
    }

    /**
     * Dosya boyutu formatı
     */
    public static function formatFileSize($bytes) {
        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        } else {
            return $bytes . ' bytes';
        }
    }

    /**
     * Güvenli dosya adı oluştur
     */
    public static function sanitizeFileName($filename) {
        // Türkçe karakterleri değiştir
        $search = ['Ç', 'Ş', 'Ğ', 'Ü', 'İ', 'Ö', 'ç', 'ş', 'ğ', 'ü', 'ı', 'ö'];
        $replace = ['C', 'S', 'G', 'U', 'I', 'O', 'c', 's', 'g', 'u', 'i', 'o'];
        $filename = str_replace($search, $replace, $filename);

        // Özel karakterleri temizle
        $filename = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $filename);

        return $filename;
    }

    /**
     * Random string oluştur
     */
    public static function generateRandomString($length = 10) {
        return bin2hex(random_bytes($length / 2));
    }

    /**
     * Pagination hesapla
     */
    public static function paginate($totalRecords, $perPage = 20, $currentPage = 1) {
        $totalPages = ceil($totalRecords / $perPage);
        $currentPage = max(1, min($currentPage, $totalPages));
        $offset = ($currentPage - 1) * $perPage;

        return [
            'total_records' => $totalRecords,
            'per_page' => $perPage,
            'current_page' => $currentPage,
            'total_pages' => $totalPages,
            'offset' => $offset,
            'has_previous' => $currentPage > 1,
            'has_next' => $currentPage < $totalPages
        ];
    }

    /**
     * Array'i Excel'den parse et
     */
    public static function parseExcelDate($value) {
        if (empty($value)) return null;

        // Excel serial date ise
        if (is_numeric($value)) {
            $unix = ($value - 25569) * 86400;
            return date('Y-m-d', $unix);
        }

        // Metin tarih ise
        return date('Y-m-d', strtotime($value));
    }

    /**
     * Excel tutarını parse et
     */
    public static function parseExcelAmount($value) {
        if (empty($value)) return 0;

        // Türkçe format: 1.234,56 → 1234.56
        $value = str_replace('.', '', $value);
        $value = str_replace(',', '.', $value);

        return floatval($value);
    }

    /**
     * Gün farkı hesapla
     */
    public static function daysDifference($date1, $date2 = null) {
        if (!$date2) {
            $date2 = date('Y-m-d');
        }

        $diff = strtotime($date1) - strtotime($date2);
        return floor($diff / (60 * 60 * 24));
    }

    /**
     * Dosya yükleme kontrolü
     */
    public static function validateUpload($file, $allowedTypes = [], $maxSize = 10485760) {
        // Dosya yüklendi mi?
        if (!isset($file) || $file['error'] !== UPLOAD_ERR_OK) {
            return ['success' => false, 'message' => 'Dosya yükleme hatası'];
        }

        // Boyut kontrolü
        if ($file['size'] > $maxSize) {
            return ['success' => false, 'message' => 'Dosya çok büyük. Maksimum: ' . self::formatFileSize($maxSize)];
        }

        // Tip kontrolü
        if (!empty($allowedTypes)) {
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($finfo, $file['tmp_name']);
            finfo_close($finfo);

            if (!in_array($mimeType, $allowedTypes)) {
                return ['success' => false, 'message' => 'Geçersiz dosya tipi'];
            }
        }

        return ['success' => true];
    }

    /**
     * JSON response
     */
    public static function jsonResponse($data, $statusCode = 200) {
        http_response_code($statusCode);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode($data, JSON_UNESCAPED_UNICODE);
        exit;
    }

    /**
     * Debug (sadece development)
     */
    public static function dd($data) {
        if (APP_DEBUG) {
            echo '<pre>';
            var_dump($data);
            echo '</pre>';
            die();
        }
    }
}
