<?php
/**
 * Database Sınıfı - PDO Singleton Pattern
 * Veritabanı bağlantı yönetimi
 */

class Database {
    private static $instance = null;
    private $connection;

    /**
     * Constructor - Private (Singleton)
     */
    private function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ];

            $this->connection = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            if (APP_DEBUG) {
                die("Veritabanı Bağlantı Hatası: " . $e->getMessage());
            } else {
                die("Veritabanı bağlantı hatası oluştu. Lütfen sistem yöneticisine başvurun.");
            }
        }
    }

    /**
     * Singleton Instance
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * PDO Connection döndür
     */
    public function getConnection() {
        return $this->connection;
    }

    /**
     * Prepared Statement çalıştır
     */
    public function query($sql, $params = []) {
        try {
            $stmt = $this->connection->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            if (APP_DEBUG) {
                die("SQL Hatası: " . $e->getMessage() . "<br>SQL: " . $sql);
            } else {
                error_log("SQL Error: " . $e->getMessage());
                return false;
            }
        }
    }

    /**
     * SELECT sorgusu çalıştır
     */
    public function select($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt ? $stmt->fetchAll() : [];
    }

    /**
     * Tek satır döndür
     */
    public function selectOne($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt ? $stmt->fetch() : null;
    }

    /**
     * INSERT sorgusu
     */
    public function insert($table, $data) {
        $fields = array_keys($data);
        $values = array_values($data);
        $placeholders = str_repeat('?,', count($fields) - 1) . '?';

        $sql = "INSERT INTO {$table} (" . implode(',', $fields) . ") VALUES ($placeholders)";

        $stmt = $this->query($sql, $values);
        return $stmt ? $this->connection->lastInsertId() : false;
    }

    /**
     * UPDATE sorgusu
     */
    public function update($table, $data, $where, $whereParams = []) {
        // Eğer $where array ise, key=value formatına çevir
        if (is_array($where)) {
            $whereParams = array_values($where);
            $whereClauses = [];
            foreach (array_keys($where) as $key) {
                $whereClauses[] = "$key = ?";
            }
            $where = implode(' AND ', $whereClauses);
        }

        $fields = array_keys($data);
        $set = implode(' = ?, ', $fields) . ' = ?';

        $sql = "UPDATE {$table} SET $set WHERE $where";

        $values = array_merge(array_values($data), $whereParams);

        $stmt = $this->query($sql, $values);
        return $stmt ? $stmt->rowCount() : false;
    }

    /**
     * DELETE sorgusu
     */
    public function delete($table, $where, $whereParams = []) {
        $sql = "DELETE FROM {$table} WHERE $where";
        $stmt = $this->query($sql, $whereParams);
        return $stmt ? $stmt->rowCount() : false;
    }

    /**
     * Transaction başlat
     */
    public function beginTransaction() {
        return $this->connection->beginTransaction();
    }

    /**
     * Transaction commit
     */
    public function commit() {
        return $this->connection->commit();
    }

    /**
     * Transaction rollback
     */
    public function rollback() {
        return $this->connection->rollBack();
    }

    /**
     * Clone ve wakeup engelle (Singleton)
     */
    private function __clone() {}
    public function __wakeup() {}
}
